import {
  ElMessage
} from 'element-plus'

import {
  getSnowflake,
  uuid,
  pointToGrid,
  equals,
  dist2d,
  angle,
  getNearestCoord,
  pointToFootDist,
  pointInSegments,
  filterPointInSegments,
  getSlope,
  collinear3Points,
  intersects,
  lineIntersect
} from './algorithm'

import nodesDefault, { gutter } from './nodesDefault'
import type { INodes, Nodes } from './INodes'

export {
  getSnowflake,
  uuid,
  pointToGrid,
  dist2d,
  angle,
  getNearestCoord,
  pointToFootDist,
  pointInSegments,
  filterPointInSegments,
  getSlope,
  collinear3Points,
  intersects,
  lineIntersect
} from './algorithm' // 默认节点属性对象

/**
 * 创建新节点
 * @author 谁改填谁
 * @param {Object} anonymous node节点对象属性
 * @returns {Object} node节点对象
 */
export function createNode({
  id = getSnowflake(), // 节点唯一标识id
  nodeType = '0',
  nodeCode = uuid(), // 用户可编辑节点code
  nodeName = '流程开始', // 节点名称
  paths = [],
  position = {
    x: 300,
    y: 20
  },
  size = {
    width: 40,
    height: 40
  }
} = {}, params = {}) {
  return {
    'id': id,
    'nodeCode': nodeCode,
    'nodeType': nodeType,
    'nodeName': nodeName,
    'nodePath': {
      position: position,
      size: size,
      /**
       * 开始节点, 结束节点 nodePath
       * demo 实际数据自己修改
       */
      // paths: createPath({
      //   position: {x: 292, y: 20},
      //   size: {width: 32, height: 32}
      // }, {
      //   position: {x: 360, y: 20},
      //   size: {width: 32, height: 32}
      // })
      paths: paths
    },
    ...params
  }
}

/**
 * 获取节点基础属性
 * @param {Object} node
 * @returns {Object} x,y,width,height,top,bottom,left,right
 */
export function getNodeBasic(node) {
  const { x, y } = node.nodePath.position
  const width = node.nodePath.size.width
  const height = node.nodePath.size.height
  return {
    x,
    y,
    width,
    height,
    top: y - height / 2,
    bottom: y + height / 2,
    left: x - width / 2,
    right: x + width / 2
  }
}

/**
 * 判断点是否在矩形内
 * @param {Array} point [100,100]
 * @param {Object} node
 * @returns {Boolean}
 */
export function inRect(point, node) {
  const { 0: x, 1: y } = point
  const { position, size } = node.nodePath
  let { x: x1, y: y1 } = position
  x1 = x1 - size.width / 2
  y1 = y1 - size.height / 2
  const x2 = x1 + node.nodePath.size.width
  const y2 = y1 + node.nodePath.size.height
  return x >= x1 && x <= x2 && y >= y1 && y <= y2
}
// 不包含边框
export function insideRect(point, node) {
  const { 0: x, 1: y } = point
  const { position, size } = node.nodePath
  let { x: x1, y: y1 } = position
  x1 = x1 - size.width / 2
  y1 = y1 - size.height / 2
  const x2 = x1 + node.nodePath.size.width
  const y2 = y1 + node.nodePath.size.height
  return x > x1 && x < x2 && y > y1 && y < y2
}
// 增加额外吸附距离
export function inRectAdsorb(point, node) {
  const adsorb = 10 * 2
  const { 0: x, 1: y } = point
  const { position, size } = node.nodePath
  let { x: x1, y: y1 } = position
  const halfWidth = (size.width + adsorb) / 2
  const halfHeight = (size.height + adsorb) / 2
  const x2 = x1 + halfWidth
  const y2 = y1 + halfHeight
  x1 = x1 - halfWidth
  y1 = y1 - halfHeight
  return x >= x1 && x <= x2 && y >= y1 && y <= y2
}

// 2个点是否在水平线或垂直线上
export function isVertical(p1, p2) {
  const { 0: x1, 1: y1 } = p1
  const { 0: x2, 1: y2 } = p2
  return x1 === x2 || y1 === y2
}

/**
 * actives多选，选中状态处理
 * 主要用在组合按键，Shift+鼠标点击
 * @param {Array} actives actives原型数组
 * @param {String} id 点击节点id
 */
export function checkActives(actives, id) {
  const index = actives.indexOf(id)
  if (index > -1) {
    actives.splice(index, 1)
  } else {
    actives.push(id)
  }
}

/**
 * 计算画布中坐标
 * @param {Number} pageX e.pageX
 * @param {Number} pageY e.pageX
 * @param {Number} oldPageX mousedown的pageX
 * @param {Number} oldPageY mousedown的pageY
 * @param {Number} x 画布的水平偏移
 * @param {Number} y 画布的垂直偏移
 * @param {Number} oldX mousedown的this.x
 * @param {Number} oldY mousedown的this.y
 * @param {Number} scale this.scale
 * @param {Number} oldScale mousedown时的this.scale
 * @param {Number} diffX 补x
 * @param {Number} diffY 补y
 */
export function getMovingPoint({
  pageX = 0,
  pageY = 0,
  oldPageX = 0,
  oldPageY = 0,
  x = 0,
  y = 0,
  oldX = 0,
  oldY = 0,
  scale = 1,
  oldScale = 1,
  diffX = 0,
  diffY = 0
} = {}) {
  return [
    pointToGrid((pageX - oldPageX - (x - oldX)) / scale + diffX),
    pointToGrid((pageY - oldPageY - (y - oldY)) / scale + diffY)
  ]
}

/**
 * 计算默认节点在画布中的中心坐标
 * @param {Array} nodeConfig
 * @param {Number} x
 * @param {Number} y
 * @returns {Object} {x,y}
 */
export function getCenterCoords(nodeConfig, svgWrap, x, y) {
  if (x !== 0) {
    x = 0
  }
  if (y !== 0) {
    y = 0
  }
  let startNodesX = nodeConfig.find(item => item.nodeType === '1').nodePath.position.x
  let startNodesY = nodeConfig.find(item => item.nodeType === '1').nodePath.position.y
  let pageOffsetWidth = svgWrap.width.baseVal.value
  return {
    x: x + 0.5 * pageOffsetWidth - startNodesX,
    y: y + 80 - startNodesY
  }
}


/**
 * 画布相对所有节点中心点
 * @param {Array} nodeConfig 数据
 * @returns {Object} center {scale,x,y}
 */
export function getCenter(nodeConfig, svgWrap) {
  const pageOffsetWidth = svgWrap.width.baseVal.value
  const pageOffsetHeight = svgWrap.height.baseVal.value
  const coordsX: number[] = []
  const coordsY: number[] = []
  // 折叠节点的嵌套子节点 / 顶层以外节点
  // let nodes = JSON.parse(JSON.stringify(nodeConfig)).filter(node => !node.foldNest && !node.exceptTop)
  let nodes = nodeConfig.filter(node => !node.foldNest && !node.exceptTop)
  nodes.forEach(item => {
    const position = item.position
    const size = item.size
    coordsX.push(position.x - size.width / 2)
    coordsX.push(position.x + size.width / 2)
    coordsY.push(position.y - size.height / 2)
    coordsY.push(position.y + size.height / 2)
  })
  const minX = Math.min.apply(null, coordsX)
  const minY = Math.min.apply(null, coordsY)
  const maxX = Math.max.apply(null, coordsX)
  const maxY = Math.max.apply(null, coordsY)
  const width = maxX - minX
  const height = maxY - minY
  const center = {
    x: maxX - width / 2,
    y: maxY - height / 2
  }
  const scale = Math.min((pageOffsetWidth - 80) / width, (pageOffsetHeight - 80) / height) // 各边留20px
  if (scale > 1) {
    let startNodesY = nodes[0].position.y
    return {
      scale: 1,
      x: 0.5 * pageOffsetWidth - center.x,
      y: 120 - startNodesY
    }
  }
  return {
    scale,
    // x: pageOffsetWidth / 2 + (width / 2 - center.x) * scale - width / 2 * scale,
    // y: pageOffsetHeight / 2 + (height / 2 - center.y) * scale - height / 2 * scale
    x: (pageOffsetWidth) / 2 - (minX + width / 2) * scale,
    y: (pageOffsetHeight) / 2 - (minY + height / 2) * scale
  }
}

// export function getCenter(nodes, svgWrap) {
//   const pageOffsetWidth = svgWrap.width.baseVal.value
//   const pageOffsetHeight = svgWrap.height.baseVal.value
//   const { width, height } = nodes.reduce((a, b) => {
//     console.log(b.size.boxHeight)
//     return {
//       width: a.width + b.size.boxWidth,
//       height: a.height + b.size.boxHeight
//     }
//   }, { width: 0, height: 0 })
//   console.log(width, pageOffsetWidth)
//   console.log(height, pageOffsetHeight)

//   const minX = nodes[0].position.left
//   const minY = nodes[0].position.top
//   // const maxX = nodes[0].position.left + width
//   // const maxY = nodes[0].position.top + height
//   const center = {
//     x: minX - width / 2,
//     y: minY - height / 2
//   }
//   const scale = Math.min(pageOffsetWidth / width, pageOffsetHeight / height) // 各边留20px
//   if (scale > 1) {
//     let startNodesY = nodes[0].position.top
//     return {
//       scale: 1,
//       x: 0.5 * pageOffsetWidth - center.x,
//       y: 80 - startNodesY
//     }
//   }
//   return {
//     scale,
//     x: (pageOffsetWidth) / 2 - (minX + width / 2) * scale,
//     y: (pageOffsetHeight) / 2 - (minY + height / 2) * scale
//   }
// }


/**
 * 生成 node 节点的每个坐标
 * @param {Array} treeNodes 树形节点数据
 * @param {Array} parent 树形节点数据
 * @returns {Nodes[]}
 */
export function getNodeConfigCoords(treeNodes: Nodes[], parent: Nodes | null = null, level = 0, heights: number[] = []) {
  heights[level] = treeNodes
    .filter(item => !['2'].includes(item.nodeType) && !item.lowerLevelNum && !item.fold)
    .reduce((a, b) => Math.max(a, nodesDefault[b.nodeType].size.height), nodesDefault[treeNodes[0].nodeType].size.height)
  return treeNodes.map((item, index, nodes) => {
    const lowerLevelNum = Number(item.lowerLevelNum ?? 0)
    const temLevel = level + lowerLevelNum
    item.level = parent?.level ? parent.level + 1 + lowerLevelNum : temLevel // 层级
    // item.fold = item.fold ?? false // 折叠状态
    const node = nodesDefault[item.nodeType] // 默认节点数据
    // 初始大小
    item.size = {
      width: node.size.width,
      height: node.size.height,
      boxWidth: node.size.width + gutter.x * 2,
      boxHeight: node.size.height + gutter.y * 2
      // exceptLowerBoxWidth: node.size.width + gutter.x * 2
    }
    // 记录各个 level 的节点 最大高度
    // if (!['2'].includes(item.nodeType) && !lowerLevelNum && !item.fold) {
    //   heights[level] = Math.max(heights[level] ?? item.size.height, item.size.height)
    // }
    // -----------
    // 初始定位 开始
    // -----------
    // item.position = {
    //   left: 0,
    //   top: item.size.boxHeight * item.level,
    //   x: item.size.boxWidth / 2,
    //   y: item.size.boxHeight * item.level + item.size.height / 2
    // }
    item.position = {}
    item.position.left = 0
    item.position.top = (parent?.position?.top ?? 0) + (parent?.size?.height ?? 0) + gutter.y * 2
    // 处理下增节点 top
    if (lowerLevelNum) {
      item.position.top += heights.slice(level, temLevel).reduce((a, b) => a + b) + lowerLevelNum * gutter.y * 2
    }
    item.position.x = item.position.left + item.size.boxWidth / 2
    item.position.y = item.position.top + item.size.height / 2
    // -----------
    // 初始定位 结束
    // -----------
    // 提前求得left，子节点累加用
    if (parent) {
      item.position.left = parent.position.left
      /**
       * 取当前节点,左侧同级节点宽度
       * 过滤助理节点 和 level不同的下增节点
       */
      const sliceNodes = nodes.slice(0, index).filter(p => String(p.nodeType) !== '5')
      item.position.left += sliceNodes.reduce((a, b) => a + b.size.boxWidth, 0)
      if (item.lowerLevelNum && index === nodes.length - 1) {
        // 下增节点处理 left 和 x
        // item.position.left = parent.position.left + parent.size.boxWidth
        // item.position.x = item.position.left + item.size.boxWidth / 2
      } else {
        // 普通节点
        // console.log(parent.nodeName, item.nodeName)
        // parent.size.boxWidth += item.size.boxWidth
        // parent.position.x = parent.size.boxWidth / 2
      }
    }
    const hasChildren = Array.isArray(item.children) && !!item.children.length
    // 开始处理子节点
    if (hasChildren && !item.fold) {
      const filterChildren = item.children?.filter(child => !['5'].includes(String(child.nodeType)) && !child.lowerLevelNum)
      if (filterChildren?.length) {
        item.size.boxWidth = 0
      }
      // -------
      // 开始递归
      // -------
      item.children = getNodeConfigCoords(item.children!, item, item.level + 1, heights)
      // item.size.boxWidth = filterChildren.reduce((a, b) => a + b.size.boxWidth, 0)
      // const filterLowerLevelChildren = filterChildren.filter((child, index) => !child.lowerLevelNum && filterChildren.length - 1 === index)
      // item.size.exceptLowerBoxWidth = filterLowerLevelChildren.reduce((a, b) => a + b.size.boxWidth, 0)
      // if (!item.size.exceptLowerBoxWidth) {
      //   item.size.exceptLowerBoxWidth = item.size.boxWidth
      // }
      // }
      // 合高
      if (parent) {
        parent.size.boxHeight += item.size.boxHeight
      }
      // console.log(heights)
      // 父节点 相对 直属子节点 水平居中
      // item.position.x = item.children.reduce((a, b) => a + b.position.x, 0) / item.children.length
    } else {
      // 父节点 相对 直属子节点 水平居中
      // item.position.x = item.position.left + item.size.boxWidth / 2
    }
    if (parent) {
      // item.position.left = parent.position.left
      /**
       * 取当前节点,左侧同级节点宽度
       * 过滤助理节点 和 level不同的下增节点
       */
      // const sliceNodes = nodes.slice(0, index).filter(p => String(p.nodeType) !== '5')
      // item.position.left += sliceNodes.reduce((a, b) => a + b.size.boxWidth, 0)
      const filterNodes = nodes.filter(p => String(p.nodeType) !== '5')
      if (item.lowerLevelNum && index === filterNodes.length - 1) {
        // console.log(item.lowerLevelNum, index, filterNodes.length - 1)
        // 下增节点处理 left 和 x，最后一个下增节点
        item.position.left = parent.position.left + parent.size.boxWidth
      } else {
        // 普通节点
        // console.log(parent.nodeName, item.nodeName)
        parent.size.boxWidth += item.size.boxWidth
        parent.position.x = parent.size.boxWidth / 2
      }
    }
    // 父节点 相对 所有子节点 绝对水平居中，效果等同于钉钉流程图
    item.position.x = item.position.left + item.size.boxWidth / 2
    // item.position.x = item.position.left + (item.size.exceptLowerBoxWidth || item.size.boxWidth) / 2
    // console.log(item.size.exceptLowerBoxWidth, item.size.boxWidth)
    // 处理 子节点包含下增节点
    // if (hasChildren && item.children.some(child => child.lowerLevelNum > 0)) {
    //   console.log(item.size.exceptLowerBoxWidth)
    // }
    // 清除末级节点 children
    if (Array.isArray(item.children) && !item.children.length || item.fold) {
      delete item.paths
    }
    // 父节点 向下处理子节点的 特殊属性和连线
    if (hasChildren && !item.fold) {
      item.children?.forEach((child, index) => {
        // 助理
        if (String(child.nodeType) === '5') {
          child.position.left = item.position.x + item.size.width * 0.1
          child.position.x = child.position.left + child.size.boxWidth / 2
          child.position.y = item.position.top + item.size.height + gutter.y * 0.5
          child.position.top = child.position.y - child.size.height / 2
        }
      })
      // 画线
      item.paths = item.children?.map(child => {
        // 助理
        const isAssistant = String(child.nodeType) === '5'
        if (isAssistant) {
          return {
            points: [
              [item.position.x, item.position.top + item.size.height],
              [item.position.x, (item.position.top + item.size.height) + gutter.y * 0.5],
              [child.position.left + gutter.x, (item.position.top + item.size.height) + gutter.y * 0.5]
            ]
          }
        }
        return {
          points: [
            [item.position.x, item.position.top + item.size.height],
            [item.position.x, (item.position.top + item.size.height) + gutter.y * 1],
            [child.position.x, (item.position.top + item.size.height) + gutter.y * 1],
            [child.position.x, child.position.top]
          ]
        }
      })
    }
    return item
  })
}

/**
 * 生成紧凑型树状图算法
 */
export function generateCompactTreeAlgorithm() {
  return {
    getNodeConfigCoords
  }
}