class TidyLayout {
  parent_child_margin: any
  peer_margin: any
  is_layered: boolean
  depth_to_y: never[]
  constructor(parent_child_margin, peer_margin, is_layered = false) {
    this.parent_child_margin = parent_child_margin
    this.peer_margin = peer_margin
    this.is_layered = is_layered
    this.depth_to_y = []
  }

  separate(node, childIndex, yList) {
    let left = new Contour(false, node.children[childIndex - 1])
    let right = new Contour(true, node.children[childIndex])

    while (!left.isNone() && !right.isNone()) {
      if (left.bottom() > yList.bottom()) {
        let top = yList.pop()
        if (top === null) {
          console.log('Err\n\n' + node.str() + '\n\nleft.bottom=' + left.bottom() + '\nyList.bottom=' + yList.bottom())
        }
        yList = top
      }

      let dist = left.right() - right.left() + this.peer_margin
      if (dist > 0) {
        right.modifierSum += dist
        this.moveSubtree(node, childIndex, yList.index, dist)
      }

      let leftBottom = left.bottom()
      let rightBottom = right.bottom()
      if (leftBottom <= rightBottom) {
        left.next()
      }
      if (leftBottom >= rightBottom) {
        right.next()
      }
    }

    if (left.isNone() && !right.isNone()) {
      this.setLeftThread(node, childIndex, right.node(), right.modifierSum)
    } else if (!left.isNone() && right.isNone()) {
      this.setRightThread(node, childIndex, left.node(), left.modifierSum)
    }

    return yList
  }

  setLeftThread(node, currentIndex, target, modifier) {
    let first = node.children[0]
    let current = node.children[currentIndex]
    let diff = modifier - first.modifierExtremeLeft - first.modifierToSubtree
    first.extremeLeft.tidy.threadLeft = target
    first.extremeLeft.tidy.modifierThreadLeft = diff
    current.tidy.extremeLeft = node.children[currentIndex].tidy.extremeLeft
    current.tidy.modifierExtremeLeft = current.tidy.modifierExtremeLeft + current.tidy.modifierToSubtree - first.tidy.modifierToSubtree
  }

  setRightThread(node, currentIndex, target, modifier) {
    let current = node.children[currentIndex]
    let diff = modifier - current.modifierExtremeRight - current.modifierToSubtree
    current.extremeRight.tidy.threadRight = target
    current.extremeRight.tidy.modifierThreadRight = diff
    let prev = node.children[currentIndex - 1].tidy
    current.tidy.extremeRight = prev.extremeRight
    current.tidy.modifierExtremeRight = prev.modifierExtremeRight + prev.modifierToSubtree - current.tidy.modifierToSubtree
  }

  moveSubtree(node, currentIndex, fromIndex, distance) {
    let child = node.children[currentIndex]
    let childTidy = child.tidy
    childTidy.modifierToSubtree += distance

    if (fromIndex !== currentIndex - 1) {
      let indexDiff = currentIndex - fromIndex
      node.children[fromIndex + 1].tidy.shiftAcceleration += distance / indexDiff
      node.children[currentIndex].tidy.shiftAcceleration -= distance / indexDiff
      node.children[currentIndex].tidy.shiftChange -= distance - distance / indexDiff
    }
  }

  setYRecursive(root) {
    if (!this.is_layered) {
      root.preOrderTraversal(node => {
        this.setY(node)
      })
    } else {
      let depthToY: any = this.depth_to_y
      depthToY.length = 0
      let margin = this.parent_child_margin
      root.bfsTraversalWithDepth((node, depth) => {
        while (depth >= depthToY.length) {
          depthToY.push(0)
        }

        if (node.parent === null || depth === 0) {
          node.y = 0
          return
        }

        let parent = node.parent
        depthToY[depth] = Math.max(depthToY[depth], depthToY[depth - 1] + parent.height + this.parent_child_margin)
      })
      root.preOrderTraversalWithDepth((node, depth) => {
        node.y = depthToY[depth]
      })
    }
  }

  setY(node) {
    node.y = node.parent ? node.parent.bottom + this.parent_child_margin : 0
  }

  firstWalk(node) {
    if (node.children.length === 0) {
      node.setExtreme()
      return
    }

    this.firstWalk(node.children[0])
    let yList = new LinkedYList(0, node.children[0].extremeRight.bottom)
    for (let i = 1; i < node.children.length; i++) {
      let currentChild = node.children[i]
      this.firstWalk(currentChild)
      let max_y = currentChild.extremeLeft.bottom()
      yList = this.separate(node, i, yList)
      yList = yList.update(i, max_y)
    }

    node.positionRoot()
    node.setExtreme()
  }

  firstWalkWithFilter(node, set) {
    if (!set.has(node)) {
      invalidateExtremeThread(node)
      return
    }

    if (node.children.length === 0) {
      node.setExtreme()
      return
    }

    this.firstWalkWithFilter(node.children[0], set)
    let yList = new LinkedYList(0, node.children[0].extremeRight.bottom)
    for (let i = 1; i < node.children.length; i++) {
      let currentChild = node.children[i]
      currentChild.tidy.modifierToSubtree = -currentChild.relative_x
      this.firstWalkWithFilter(currentChild, set)
      let max_y = currentChild.extremeLeft.bottom()
      yList = this.separate(node, i, yList)
      yList = yList.update(i, max_y)
    }

    node.positionRoot()
    node.setExtreme()
  }

  secondWalk(node, modSum) {
    modSum += node.tidy.modifierToSubtree
    node.x = node.relative_x + modSum
    node.addChildSpacing()

    for (let child of node.children) {
      this.secondWalk(child, modSum)
    }
  }

  secondWalkWithFilter(node, modSum, set) {
    modSum += node.tidy.modifierToSubtree
    let newX = node.relative_x + modSum
    if (Math.abs(newX - node.x) < 1e-8 && !set.has(node)) {
      return
    }

    node.x = newX
    node.addChildSpacing()

    for (let child of node.children) {
      this.secondWalkWithFilter(child, modSum, set)
    }
  }

  layout(root) {
    root.preOrderTraversal(initNode)
    this.setYRecursive(root)
    this.firstWalk(root)
    this.secondWalk(root, 0)
  }

  partialLayout(root, changed) {
    if (this.is_layered) {
      this.layout(root)
      return
    }

    for (let node of changed) {
      if (!node.tidy) {
        initNode(node)
      }
      this.setYRecursive(node)
    }

    let set = new Set(changed)
    for (let node of changed) {
      set.add(node)
      let current = node
      while (current.parent !== null) {
        invalidateExtremeThread(current)
        set.add(current.parent)
        current = current.parent
      }
    }

    this.firstWalkWithFilter(root, set)
    this.secondWalkWithFilter(root, 0, set)
  }
}

class Contour {
  modifierSum: any
  isLeft: any
  current: any
  constructor(isLeft, current) {
    this.isLeft = isLeft
    this.current = current
    this.modifierSum = current.tidy.modifierToSubtree
  }

  node() {
    return this.current
  }

  isNone() {
    return this.current === null
  }

  left() {
    let node = this.node()
    return this.modifierSum + node.relative_x - node.width / 2
  }

  right() {
    let node = this.node()
    return this.modifierSum + node.relative_x + node.width / 2
  }

  bottom() {
    return this.current ? this.current.y + this.current.height : 0
  }

  next() {
    if (this.current) {
      let node = this.node()
      if (this.isLeft) {
        if (node.children.length > 0) {
          this.current = node.children[0]
          this.modifierSum += node.children[0].tidy.modifierToSubtree
        } else {
          this.modifierSum += node.tidy.modifierThreadLeft
          this.current = node.tidy.threadLeft
        }
      } else if (node.children.length > 0) {
        this.current = node.children[node.children.length - 1]
        this.modifierSum += node.children[node.children.length - 1].tidy.modifierToSubtree
      } else {
        this.modifierSum += node.tidy.modifierThreadRight
        this.current = node.tidy.threadRight
      }
    }
  }
}

class TreeNode {
  id: any
  width: any
  height: any
  parent: null
  children: any[]
  x: number
  y: number
  relative_x: number
  relative_y: number
  tidy: any
  constructor(id, width, height) {
    this.id = id
    this.width = width
    this.height = height
    this.parent = null
    this.children = []
    this.x = 0
    this.y = 0
    this.relative_x = 0
    this.relative_y = 0
    this.tidy = null
  }

  appendChild(child) {
    this.children.push(child)
    child.parent = this
  }

  preOrderTraversal(callback) {
    callback(this)
    for (let child of this.children) {
      child.preOrderTraversal(callback)
    }
  }

  preOrderTraversalWithDepth(callback, depth = 0) {
    callback(this, depth)
    for (let child of this.children) {
      child.preOrderTraversalWithDepth(callback, depth + 1)
    }
  }

  bfsTraversalWithDepth(callback, depth = 0) {
    callback(this, depth)
    for (let child of this.children) {
      child.bfsTraversalWithDepth(callback, depth + 1)
    }
  }

  setExtreme() {
    if (this.children.length === 0) {
      this.tidy = {
        extremeLeft: this,
        extremeRight: this,
        modifierToSubtree: 0,
        modifierExtremeLeft: 0,
        modifierExtremeRight: 0,
        threadLeft: null,
        threadRight: null,
        modifierThreadLeft: 0,
        modifierThreadRight: 0,
        shiftAcceleration: 0,
        shiftChange: 0
      }
    } else {
      let first = this.children[0].tidy
      this.tidy = {
        extremeLeft: first.extremeLeft,
        extremeRight: this.children[this.children.length - 1].tidy.extremeRight,
        modifierToSubtree: first.modifierToSubtree + first.modifierExtremeLeft,
        modifierExtremeLeft: first.modifierToSubtree + first.modifierExtremeLeft,
        modifierExtremeRight: this.children[this.children.length - 1].tidy.modifierToSubtree + this.children[this.children.length - 1].tidy.modifierExtremeRight,
        threadLeft: null,
        threadRight: null,
        modifierThreadLeft: 0,
        modifierThreadRight: 0,
        shiftAcceleration: 0,
        shiftChange: 0
      }
    }
  }

  positionRoot() {
    let firstChild = this.children[0]
    let firstChildPos = firstChild.relative_x + firstChild.tidy.modifierToSubtree
    let lastChild = this.children[this.children.length - 1]
    let lastChildPos = lastChild.relative_x + lastChild.tidy.modifierToSubtree
    this.relative_x = (firstChildPos + lastChildPos) / 2
    this.tidy.modifierToSubtree = -this.relative_x
  }

  addChildSpacing() {
    let speed = 0
    let delta = 0
    for (let child of this.children) {
      let childTidy = child.tidy
      speed += childTidy.shiftAcceleration
      delta += speed + childTidy.shiftChange
      childTidy.modifierToSubtree += delta
      childTidy.shiftAcceleration = 0
      childTidy.shiftChange = 0
    }
  }

  extremeLeft() {
    return this.tidy.extremeLeft
  }

  extremeRight() {
    return this.tidy.extremeRight
  }

  bottom() {
    return this.y + this.height
  }
}

function initNode(node) {
  if (node.tidy) {
    let tidy = node.tidy
    tidy.extremeLeft = null
    tidy.extremeRight = null
    tidy.shiftAcceleration = 0
    tidy.shiftChange = 0
    tidy.modifierToSubtree = 0
    tidy.modifierExtremeLeft = 0
    tidy.modifierExtremeRight = 0
    tidy.threadLeft = null
    tidy.threadRight = null
    tidy.modifierThreadLeft = 0
    tidy.modifierThreadRight = 0
  } else {
    node.tidy = {
      extremeLeft: null,
      extremeRight: null,
      shiftAcceleration: 0,
      shiftChange: 0,
      modifierToSubtree: 0,
      modifierExtremeLeft: 0,
      modifierExtremeRight: 0,
      threadLeft: null,
      threadRight: null,
      modifierThreadLeft: 0,
      modifierThreadRight: 0
    }
  }

  node.x = 0
  node.y = 0
  node.relative_x = 0
  node.relative_y = 0
}

function invalidateExtremeThread(node) {
  node.setExtreme()
  let eLeft = node.extremeLeft().tidy
  eLeft.threadLeft = null
  eLeft.threadRight = null
  eLeft.modifierThreadLeft = 0
  eLeft.modifierThreadRight = 0
  let eRight = node.extremeRight().tidy
  eRight.threadLeft = null
  eRight.threadRight = null
  eRight.modifierThreadLeft = 0
  eRight.modifierThreadRight = 0
}

class LinkedYList {
  index: any
  bottom: any
  next: any
  constructor(index, bottom) {
    this.index = index
    this.bottom = bottom
    this.next = null
  }

  pop() {
    return this.next
  }

  update(index, bottom) {
    let node = this
    while (node.index !== index && node.next !== null) {
      node = node.next
    }
    node.next = new LinkedYList(index, bottom)
    return this
  }
}

class SetUsize extends Set {
  constructor() {
    super()
  }

  insert(element) {
    super.add(element)
  }

  contains(element) {
    return super.has(element)
  }
}

module.exports = {
  TidyLayout,
  Node: TreeNode
}
