import { ref, type Ref } from 'vue'
import { type RouteLocationNormalizedGeneric } from 'vue-router'
import { defineStore } from 'pinia'
import { type AxiosResponse } from 'axios'
import { to } from 'await-to-js'
// import defAva from '@/assets/images/profile.jpg'
import logo from '@/assets/logo.png'
import store from '@/store'
import { login as loginApi, logout as logoutApi, getInfo as getUserInfo } from '@/api/login'
import { ssoEnable, authorize as authorizeApi, ssoLogin as ssoLoginApi, logoutUrl as logoutUrlAPI } from '@/api/oauth2'
import Tab from '@/plugins/tab'
import { getToken, removeToken, setToken } from '@/utils/auth'
// import { LoginData } from '@/api/types'

export const useUserStore = defineStore('user', (): {
  sso: Ref<boolean>
  userId: Ref<string | number>
  userInfo: Ref<any>
  token: Ref<null | string>
  passwordStatus: Ref<string>
  nickname: Ref<string>
  avatar: Ref<string>
  roles: Ref<Array<string>>
  permissions: Ref<Array<string>>
  lastRoute: Ref<string>
  updateSso: () => Promise<void>
  authorize: (route: RouteLocationNormalizedGeneric) => Promise<AxiosResponse['data']>
  ssoLogin: (query: any) => Promise<void>
  login: (userInfo: any) => Promise<void>
  getInfo: () => Promise<void>
  logout: () => Promise<void>
  ssoLogout: () => Promise<AxiosResponse['data']>
  logoutFeed: () => Promise<void>
  setAvatar: (value: string) => void
} => {
  const sso = ref<boolean>(true)
  const token = ref(getToken())
  const passwordStatus = ref('')
  const name = ref('')
  const nickname = ref('')
  const userId = ref<string | number>('')
  const avatar = ref('')
  const roles = ref<Array<string>>([]) // 用户角色编码集合 → 判断路由权限
  const permissions = ref<Array<string>>([]) // 用户权限编码集合 → 判断按钮权限
  const userInfo = ref({})
  const lastRoute = ref(String(sessionStorage.getItem('__LAST_ROUTE__') || ''))

  const updateSso = (): Promise<void> => {
    return new Promise((resolve) => {
      ssoEnable().then(res => {
        sso.value = res.data
        resolve()
      })
    })
  }
  /**
   * 单点登录 - 获取登录地址
   * @param code
   * @returns
   */
  const authorize = async (route: RouteLocationNormalizedGeneric): Promise<AxiosResponse['data']> => {
    // console.log('authorize')
    const [err, res] = await to(authorizeApi())
    if (res) {
      const url = res as AxiosResponse['data']
      // console.log(url + '&state=' + route.fullPath)
      window.location.replace(url + '&state=' + route.fullPath)
    }
    return Promise.reject(err)
  }

  /**
   * 单点登录
   * @param {object} query
   * @param {string} query.code code
   * @returns
   */
  const ssoLogin = async (query): Promise<void> => {
    const [err, res] = await to(ssoLoginApi(query))
    if (res) {
      const { data } = res as AxiosResponse['data']
      setToken(data.access_token)
      token.value = data.access_token
      return Promise.resolve(data)
    }
    return Promise.reject(err)
  }

  /**
   * 登录
   * @param userInfo
   * @returns
   */
  const login = async (userInfo: any): Promise<void> => {
    console.log('login')
    const tenantId = userInfo.tenantId.trim()
    const username = userInfo.username.trim()
    const password = userInfo.password
    const code = userInfo.code
    const uuid = userInfo.uuid
    const [err, res] = await to(loginApi(tenantId, username, password, code, uuid))
    if (res) {
      const { data } = res as AxiosResponse['data']
      setToken(data.access_token)
      token.value = data.access_token
      return Promise.resolve()
    }
    return Promise.reject(err)
  }

  // 获取用户信息
  const getInfo = async (): Promise<void> => {
    const [err, res] = await to(getUserInfo())
    if (res) {
      const data = res as AxiosResponse['data']
      const user = data.user
      const profile = user.avatar === '' || user.avatar === null ? logo : user.avatar

      if (data.roles && data.roles.length > 0) {
        // 验证返回的roles是否是一个非空数组
        roles.value = data.roles
        permissions.value = data.permissions
      } else {
        roles.value = ['ROLE_DEFAULT']
      }
      passwordStatus.value = data.passwordStatus
      name.value = user.userName
      nickname.value = user.nickName
      avatar.value = profile
      userId.value = user.userId
      userInfo.value = user
      return Promise.resolve()
    }
    return Promise.reject(err)
  }

  // 注销
  const logout = async (): Promise<void> => {
    await logoutApi()
    token.value = ''
    roles.value = []
    permissions.value = []
    removeToken()
    Tab.closeAllPage()
  }
  // 注销
  const ssoLogout = async (): Promise<any> => {
    await logoutApi()
    const url = await logoutUrlAPI()
    token.value = ''
    roles.value = []
    permissions.value = []
    removeToken()
    Tab.closeAllPage()
    window.location.replace(url + '?redirect_uri=' + window.location.href)
  }

  // 注销 不请求退出登录接口
  const logoutFeed = async (): Promise<void> => {
    token.value = ''
    roles.value = []
    permissions.value = []
    removeToken()
    Tab.closeAllPage()
  }

  const setAvatar = (value: string) => {
    avatar.value = value
  }

  return {
    sso,
    userId,
    userInfo,
    token,
    passwordStatus,
    nickname,
    avatar,
    roles,
    permissions,
    lastRoute,
    updateSso,
    authorize,
    ssoLogin,
    login,
    getInfo,
    logout,
    ssoLogout,
    logoutFeed,
    setAvatar
  }
})

export default useUserStore
// 非setup
export function useUserStoreHook() {
  return useUserStore(store)
}
