package watcher;

import handler.FileHandlerFactory;
import lombok.extern.slf4j.Slf4j;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.*;
import java.util.function.Predicate;


@Slf4j
public class DirWatcher {

    private final Path dir;
    private final Predicate<File> fileFilter;
    private final FileHandlerFactory fileHandler;

    public DirWatcher(Path dir, Predicate<File> fileFilter, FileHandlerFactory fileHandler) {
        this.dir = dir;
        this.fileFilter = fileFilter;
        this.fileHandler = fileHandler;
    }

    public void startWatch() {
        Thread.ofVirtual().start(this::watchLoop);
    }

    private void watchLoop() {
        try {
            // 初始处理，将文件夹下有的文件先处理
            File[] files = dir.toFile().listFiles(fileFilter::test);
            if (files != null) {
                for (File f : files) {
                    Thread.ofVirtual().start(fileHandler.create(f));
                }
            }

            WatchService watchService = FileSystems.getDefault().newWatchService();
            dir.register(watchService, StandardWatchEventKinds.ENTRY_CREATE);
            while (true) {
                WatchKey key = watchService.take();
                for (WatchEvent<?> event : key.pollEvents()) {
                    Path fullPath = dir.resolve((Path) event.context());
                    File file = fullPath.toFile();
                    if (!fileFilter.test(file)) {
                        continue;
                    }
                    if (!waitForFileReady(file, 3, 500)) {
                        log.error("文件打开失败,已跳过:{}", file.getName());
                        continue;
                    }

                    log.info("文件已创建: {}", file.getName());
                    Thread.ofVirtual().start(fileHandler.create(file));
                }
                key.reset();
            }

        } catch (IOException | InterruptedException e) {
            log.error(e.getMessage());
        }
    }

    // 避免文件正在写入，重试机制
    private boolean waitForFileReady(File file, int maxRetries, long intervalMs) {
        for (int i = 0; i < maxRetries; i++) {
            try (FileInputStream fis = new FileInputStream(file)) {
                // 能打开就认为文件已写入完成
                return true;
            } catch (IOException e) {
                try {
                    Thread.sleep(intervalMs);
                } catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    break;
                }
            }
        }
        return false; // 超过最大重试次数仍无法读取
    }
}
