package text;

import lombok.extern.slf4j.Slf4j;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.*;
import java.util.function.Predicate;

@Slf4j
public class TextFileWatcher {

    private final Path dir;
    private final Predicate<File> fileFilter;

    public TextFileWatcher(Path dir, Predicate<File> fileFilter) {
        this.dir = dir;
        this.fileFilter = fileFilter;
    }

    public void startWatch() {
        Thread.ofVirtual().start(this::watchLoop);
    }

    private void watchLoop() {
        try {
            // 初始处理，将文件夹下有的文件先处理
            File[] files = dir.toFile().listFiles(fileFilter::test);
            if (files != null) {
                for (File f : files) {
                    TextHandler textHandler = new TextHandler(f);
                    Thread.ofVirtual().start(textHandler);
                }
            }

            WatchService watchService = FileSystems.getDefault().newWatchService();
            dir.register(watchService, StandardWatchEventKinds.ENTRY_CREATE);
            while (true) {
                WatchKey key = watchService.take();
                for (WatchEvent<?> event : key.pollEvents()) {
                    Path fullPath = dir.resolve((Path) event.context());
                    File file = fullPath.toFile();

                    if(fileFilter.test(file) && waitForFileReady(file, 3, 500)){
                        log.info("文件已创建: {}", file.getName());
                        TextHandler textHandler = new TextHandler(file);
                        Thread.ofVirtual().start(textHandler);
                    }
                }
                key.reset();
            }

        } catch (IOException | InterruptedException e) {
            log.error(e.getMessage());
        }
    }

    // 避免文件正在写入，重试机制
    private boolean waitForFileReady(File file, int maxRetries, long intervalMs) {
        for (int i = 0; i < maxRetries; i++) {
            try (FileInputStream fis = new FileInputStream(file)) {
                // 能打开就认为文件已写入完成
                return true;
            } catch (IOException e) {
                try {
                    Thread.sleep(intervalMs);
                } catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    break;
                }
            }
        }
        return false; // 超过最大重试次数仍无法读取
    }
}
